﻿unit Event_Logs;

interface

uses
{$IF DEFINED (ISFEXGUI)}
  GUI, Modules,
{$IFEND}
  ByteStream, Classes, Clipbrd, Common, Contnrs, DataEntry, DataStorage, DateUtils, DIRegex,
  EvtxReader, Graphics, PropertyList, Math, Regex, SQLite3, SysUtils, Variants, XML,
  //-----
  Artifact_Utils      in 'Common\Artifact_Utils.pas',
  Columns             in 'Common\Column_Arrays\Columns.pas',
  Icon_List           in 'Common\Icon_List.pas',
  RS_DoNotTranslate   in 'Common\RS_DoNotTranslate.pas';
  //-----

type
  TDataStoreFieldArray = array of TDataStoreField;
  PSQL_FileSearch = ^TSQL_FileSearch;

  TSQL_FileSearch = record
    fi_Carve_Header               : string;
    fi_Carve_Footer               : string;
    fi_Carve_Adjustment           : integer;
    fi_Icon_Category              : integer;
    fi_Icon_Program               : integer;
    fi_Icon_OS                    : integer;
    fi_Rgx_Itun_Bkup_Dmn          : string;
    fi_Rgx_Itun_Bkup_Nme          : string;
    fi_Name_Program               : string;
    fi_Name_Program_Type          : string;
    fi_Name_OS                    : string;
    fi_NodeByName                 : string;
    fi_Process_As                 : string;
    fi_Process_ID                 : string;
    fi_Reference_Info             : string;
    fi_Regex_Search               : string;
    fi_RootNodeName               : string;
    fi_Signature_Parent           : string;
    fi_Signature_Sub              : string;
    fi_SQLStatement               : string;
    fi_SQLTables_Required         : array of string;
    fi_SQLPrimary_Tablestr        : string;
    fi_Test_Data                  : string;
  end;

const
  ATRY_EXCEPT_STR           = 'TryExcept: '; // noslz
  ANDROID                   = 'Android'; // noslz
  BS                        = '\';
  BL_LOG_XML_DATA           = False;
  BL_PROCEED_LOGGING        = False;
  CANCELED_BY_USER          = 'Canceled by user.';
  CATEGORY_NAME             = 'Windows Event Logs';
  CHAR_LENGTH               = 80;
  COLON                     = ':';
  COMMA                     = ',';
  CR                        = #13#10;
  DCR                       = #13#10 + #13#10;
  EVENT_LOG_XML             = 'Event Log (XML)'; // noslz
  FBN_ITUNES_BACKUP_DOMAIN  = 'iTunes Backup Domain'; // noslz
  FBN_ITUNES_BACKUP_NAME    = 'iTunes Backup Name'; // noslz
  GFORMAT_STR               = '%-1s %-12s %-8s %-15s %-25s %-30s %-20s';
  GROUP_NAME                = 'Artifact Analysis';
  HYPHEN                    = ' - ';
  HYPHEN_NS                 = '-';
  IOS                       = 'iOS'; // noslz
  NUMBEROFSEARCHITEMS       = 7; //********************************************
  PLIST_BINARY_SIG          = 'Plist (Binary)'; // noslz
  PROCESSALL                = 'PROCESSALL'; // noslz
  PROCESS_AS_EVENTLOG       = 'PROCESSASEVENTLOG'; // noslz
  PROGRAM_NAME              = 'Event Logs';
  RPAD_VALUE                = 55;
  RUNNING                   = '...';
  SCRIPT_DESCRIPTION        = 'Extract Windows Event Logs';
  SCRIPT_NAME               = 'Event_Logs.pas'; // noslz
  SIG_SKYPE                 = 'Skype'; // noslz
  SIG_SQLITE                = 'SQLite'; // noslz
  SPACE                     = ' ';
  STR_FILES_BY_PATH         = 'Files by path';
  TSWT                      = 'The script will terminate.';
  USE_FLAGS_BL              = False;
  VERBOSE                   = False;
  WINDOWS                   = 'Windows'; // noslz
  XML_SIG                   = 'XML'; // noslz

  DNT_TIMECREATED           = 'DNT_TimeCreated'; // noslz

const { column setup }

  Array_Items_1: TSQL_Table_array = ( // Event Logs
{01} (sql_col: DNT_TIMECREATED;     fex_col: 'Time Created';    read_as: ftString;   convert_as: '';   col_type: ftDateTime;     show: True),
{02} (sql_col: 'DNT_Provider';      fex_col: 'Provider';        read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{03} (sql_col: 'DNT_EventRecordID'; fex_col: 'Event Record ID'; read_as: ftString;   convert_as: '';   col_type: ftInteger;      show: True),
{04} (sql_col: 'DNT_EventID';       fex_col: 'EventID';         read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{05} (sql_col: 'DNT_Level';         fex_col: 'Level';           read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{06} (sql_col: 'DNT_EventData';     fex_col: 'Event Data';      read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{07} (sql_col: 'DNT_Version';       fex_col: 'Version';         read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{08} (sql_col: 'DNT_Task';          fex_col: 'Task';            read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{09} (sql_col: 'DNT_Opcode';        fex_col: 'Opcode';          read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{10} (sql_col: 'DNT_Keywords';      fex_col: 'Keywords';        read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{11} (sql_col: 'DNT_Execution';     fex_col: 'Execution';       read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{12} (sql_col: 'DNT_Channel';       fex_col: 'Channel';         read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{13} (sql_col: 'DNT_Computer';      fex_col: 'Computer';        read_as: ftString;   convert_as: '';   col_type: ftString;       show: True),
{14} (sql_col: 'DNT_Security';      fex_col: 'Security';        read_as: ftString;   convert_as: '';   col_type: ftString;       show: True));

var
  gArr_ValidatedFiles_TList:      array [1 .. NUMBEROFSEARCHITEMS] of TList;
  gArtConnect_ProgFldr:           array [1 .. NUMBEROFSEARCHITEMS] of TArtifactConnectEntry;
  FileItems:                      array [1 .. NUMBEROFSEARCHITEMS] of PSQL_FileSearch;

  anEntry:                        TEntry;
  gArtConnect_CatFldr:            TArtifactEntry1;
  gArtifactsDataStore:            TDataStore;
  col_source_created:             TDataStoreField;
  col_source_file:                TDataStoreField;
  //col_source_location:          TDataStoreField;
  col_source_modified:            TDataStoreField;
  col_source_path:                TDataStoreField;
  current_str:                    string;
  Display_str:                    string;
  gFileSystemDataStore:           TDataStore;
  gtick_doprocess_i64:            uint64;
  gtick_doprocess_str:            string;
  gtick_foundlist_i64:            uint64;
  gtick_foundlist_str:            string;
  gtotal_validated_files_int:     integer;
  n, r:                           integer;
  param:                          string;
  gParameter_Num_StringList:       TStringList;
  previous_str:                   string;
  process_all_bl:                 boolean;
  process_proceed_bl:             boolean;
  progress_program_str:           string;
  Ref_Num:                        integer;
  regex_search_str:               string;
  s:                              integer;
  temp_int:                       integer;
  temp_process_counter:           integer;
  test_param_int:                 integer;

function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
function FileSubSignatureMatch(anEntry: TEntry): boolean;
function GetFullName(Item: PSQL_FileSearch): string;
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
function RPad(const AString: string; AChars: integer): string;
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
function TestForDoProcess(ARefNum: integer): boolean;
function TotalValidatedFileCountInTLists: integer;
procedure DetermineThenSkipOrAdd(anEntry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; Reference_Number: integer; aItems: TSQL_Table_array);
procedure LoadFileItems;

{$IF DEFINED (ISFEXGUI)}

type
  TScriptForm = class(TObject)
  private
    frmMain: TGUIForm;
    FMemo: TGUIMemoBox;
    pnlBottom: TGUIPanel;
    btnOK: TGUIButton;
    btnCancel: TGUIButton;
  public
    ModalResult: boolean;
    constructor Create;
    function ShowModal: boolean;
    procedure OKClick(Sender: TGUIControl);
    procedure CancelClick(Sender: TGUIControl);
    procedure SetText(const Value: string);
    procedure SetCaption(const Value: string);
  end;
{$IFEND}

implementation

{$IF DEFINED (ISFEXGUI)}

constructor TScriptForm.Create;
begin
  inherited Create;
  frmMain := NewForm(nil, SCRIPT_NAME);
  frmMain.Size(500, 480);
  pnlBottom := NewPanel(frmMain, esNone);
  pnlBottom.Size(frmMain.Width, 40);
  pnlBottom.Align(caBottom);
  btnOK := NewButton(pnlBottom, 'OK');
  btnOK.Position(pnlBottom.Width - 170, 4);
  btnOK.Size(75, 25);
  btnOK.Anchor(False, True, True, True);
  btnOK.DefaultBtn := True;
  btnOK.OnClick := OKClick;
  btnCancel := NewButton(pnlBottom, 'Cancel');
  btnCancel.Position(pnlBottom.Width - 88, 4);
  btnCancel.Size(75, 25);
  btnCancel.Anchor(False, True, True, True);
  btnCancel.CancelBtn := True;
  btnCancel.OnClick := CancelClick;
  FMemo := NewMemoBox(frmMain, [eoReadOnly]); // eoNoHScroll, eoNoVScroll
  FMemo.Color := clWhite;
  FMemo.Align(caClient);
  FMemo.FontName('Courier New');
  frmMain.CenterOnParent;
  frmMain.Invalidate;
end;

procedure TScriptForm.OKClick(Sender: TGUIControl);
begin
  // Set variables for use in the main proc. Must do this before closing the main form.
  ModalResult := False;
  ModalResult := True;
  frmMain.Close;
end;

procedure TScriptForm.CancelClick(Sender: TGUIControl);
begin
  ModalResult := False;
  Progress.DisplayMessageNow := CANCELED_BY_USER;
  frmMain.Close;
end;

function TScriptForm.ShowModal: boolean;
begin
  Execute(frmMain);
  Result := ModalResult;
end;

procedure TScriptForm.SetText(const Value: string);
begin
  FMemo.Text := Value;
end;

procedure TScriptForm.SetCaption(const Value: string);
begin
  frmMain.Text := Value;
end;
{$IFEND}

procedure LoadFileItems;
const
  lFileItems: array [1 .. NUMBEROFSEARCHITEMS] of TSQL_FileSearch = (

  ( // ~1~ All Event Logs (as a single list)
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'All';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       '\.evtx'; // noslz
  fi_Signature_Parent:   EVENT_LOG_XML;
  fi_Reference_Info:     ''),

  ( // ~2~ Application.evtx
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'Application';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       '^Application\.evtx'; // noslz
  fi_Signature_Parent:   EVENT_LOG_XML;
  fi_Reference_Info:     ''),

  ( // ~3~ Microsoft-Windows-PrintService%4Admin
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'Microsoft-Windows-PrintService%4Admin';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       '^Microsoft-Windows-PrintService\%4Admin\.evtx'; // noslz
  fi_Signature_Parent:   EVENT_LOG_XML;
  fi_Reference_Info:     ''),

  ( // ~4~ Operational.evtx
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'Operational';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       'Operational\.evtx'; // noslz
  fi_Signature_Parent:   EVENT_LOG_XML;
  fi_Reference_Info:     ''),

  ( // ~5~ Security.evtx
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'Security';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       '^Security\.evtx'; // noslz
  fi_Signature_Parent:   EVENT_LOG_XML;
  fi_Reference_Info:     ''),

  ( // ~6~ System.evtx
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'System';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       '^System\.evtx'; // noslz
  fi_Signature_Parent:   'Event Log (XML)'; // noslz
  fi_Reference_Info:     ''),

  ( // ~7~ Windows PowerShell.evtx
  fi_Process_As:         PROCESS_AS_EVENTLOG;
  fi_Name_Program:       'Windows PowerShell';
  fi_Name_Program_Type:  '.evtx';
  fi_Icon_Category:      1178;
  fi_Icon_Program:       1177;
  fi_Icon_OS:            ICON_WINDOWS;
  fi_Regex_Search:       '^Windows PowerShell\.evtx'; // noslz
  fi_Signature_Parent:   EVENT_LOG_XML;
  fi_Reference_Info:     ''),

  );

var
  iIdx: integer;
begin
  for iIdx := Low(lFileItems) to High(lFileItems) do
  begin
    FileItems[iIdx] := @lFileItems[iIdx];
    if not Progress.isRunning then
      break;
  end;
end;

// ~~~~

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// ~~~~

procedure Add40CharFiles(UniqueList: TUniqueListOfEntries);
var
  anEntry: TEntry;
  FileSystemDS: TDataStore;
begin
  FileSystemDS := GetDataStore(DATASTORE_FILESYSTEM);
  if assigned(UniqueList) and assigned(FileSystemDS) then
  begin
    anEntry := FileSystemDS.First;
    while assigned(anEntry) and Progress.isRunning do
    begin
      begin
        if RegexMatch(anEntry.EntryName, '^[0-9a-fA-F]{40}', False) and (anEntry.EntryNameExt = '') then // noslz iTunes Backup files
        begin
          UniqueList.Add(anEntry);
        end;
      end;
      anEntry := FileSystemDS.Next;
    end;
    FreeAndNil(FileSystemDS);
  end;
end;

// -----------------------------------------------------------------------------
// Column Value By Name As Date Time
// -----------------------------------------------------------------------------
function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
var
  iCol: integer;
begin
  Result := 0;
  iCol := ColumnByName(Statement, copy(colConf.sql_col, 5, Length(colConf.sql_col)));
  if (iCol > -1) then
  begin
    if colConf.read_as = ftLargeInt then
      Result := Int64ToDateTime_ConvertAs(Statement.Columnint64(iCol), colConf.convert_as)
    else if colConf.read_as = ftFloat then
      Result := GHFloatToDateTime(Statement.Columnint64(iCol), colConf.convert_as);
  end;
end;

// -----------------------------------------------------------------------------
// Determine Then Skip Or Add
// -----------------------------------------------------------------------------
procedure DetermineThenSkipOrAdd(anEntry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
var
  DeterminedFileDriverInfo: TFileTypeInformation;
  MatchSignature_str: string;
  NowProceed_bl: boolean;
  i: integer;
  File_Added_bl: boolean;
  Item: PSQL_FileSearch;
  reason_str: string;
  Reason_StringList: TStringList;
  trunc_EntryName_str: string;

begin
  File_Added_bl := False;
  if anEntry.isSystem and ((POS('$I30', anEntry.EntryName) > 0) or (POS('$90', anEntry.EntryName) > 0)) then
  begin
    NowProceed_bl := False;
    Exit;
  end;

  DeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
  reason_str := '';
  trunc_EntryName_str := copy(anEntry.EntryName, 1, 25);
  Reason_StringList := TStringList.Create;
  try
    try
      for i := 1 to NUMBEROFSEARCHITEMS do
      begin
        if not Progress.isRunning then
          break;
        NowProceed_bl := False;
        reason_str := '';
        Item := FileItems[i];

        // ---------------------------------------------------------------------
        // Special validation (these files are still sent here via the Regex match)
        // ---------------------------------------------------------------------
        // Proceed if SubDriver has been identified
        if (not NowProceed_bl) then
        begin
          if Item^.fi_Signature_Sub <> '' then
          begin
            if RegexMatch(RemoveSpecialChars(DeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item^.fi_Signature_Sub), False) then
            begin
              NowProceed_bl := True;
              reason_str := 'ShortDisplayName = Required SubSig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + Item^.fi_Signature_Sub + ')';
              Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(A)', IntToStr(i), IntToStr(anEntry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
            end;
          end;
        end;

        // Set the MatchSignature to the parent
        if (not NowProceed_bl) and not(UpperCase(anEntry.Extension) = '.JSON') then
        begin
          MatchSignature_str := '';
          if Item^.fi_Signature_Sub = '' then
            MatchSignature_str := UpperCase(Item^.fi_Signature_Parent);
          // Proceed if SubDriver is blank, but File/Path Name and Parent Signature match
          if ((RegexMatch(anEntry.EntryName, Item^.fi_Regex_Search, False)) or (RegexMatch(anEntry.FullPathName, Item^.fi_Regex_Search, False))) and
            ((UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) or (RegexMatch(DeterminedFileDriverInfo.ShortDisplayName, MatchSignature_str, False))) then
          begin
            NowProceed_bl := True;
            reason_str := 'ShortDisplay matches Parent sig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
            Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(B)', IntToStr(i), IntToStr(anEntry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
          end;
        end;

        // Proceed if EntryName is unknown, but iTunes Domain, Name and Sig match
        if (not NowProceed_bl) then
        begin
          if RegexMatch(biTunes_Domain_str, Item^.fi_Rgx_Itun_Bkup_Dmn, False) and RegexMatch(biTunes_Name_str, Item^.fi_Rgx_Itun_Bkup_Nme, False) and
            (UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) then
          begin
            NowProceed_bl := True;
            reason_str := 'Proceed on Sig and iTunes Domain\Name:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
            Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(C)', IntToStr(i), IntToStr(anEntry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
          end;
        end;

        if NowProceed_bl then
        begin
          gArr_ValidatedFiles_TList[i].Add(anEntry);
          File_Added_bl := True;
          if USE_FLAGS_BL then
            anEntry.Flags := anEntry.Flags + [Flag5]; // Green Flag
        end;
      end;

      if NOT(File_Added_bl) then
        Reason_StringList.Add(format(GFORMAT_STR, ['', 'Ignored', '', IntToStr(anEntry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]));

    finally
      for i := 0 to Reason_StringList.Count - 1 do
        Progress.Log(Reason_StringList[i]);
    end;

  finally
    Reason_StringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// File Sub Signature Match
// -----------------------------------------------------------------------------
function FileSubSignatureMatch(anEntry: TEntry): boolean;
var
  i: integer;
  param_num_int: integer;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  Item: PSQL_FileSearch;
begin
  Result := False;
  if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
  begin
    for i := 1 to NUMBEROFSEARCHITEMS do
    begin
      if not Progress.isRunning then
        break;
      Item := FileItems[i];
      if Item^.fi_Signature_Sub <> '' then
      begin
        aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
        if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item^.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
        begin
          if BL_PROCEED_LOGGING then
            Progress.Log(RPad('Proceed' + HYPHEN + 'Identified by SubSig:', RPAD_VALUE) + anEntry.EntryName + SPACE + 'Bates:' + IntToStr(anEntry.ID));
          Result := True;
          break;
        end;
      end;
    end;
  end
  else
  begin
    if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
    begin
      for i := 0 to gParameter_Num_StringList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        param_num_int := StrToInt(gParameter_Num_StringList[i]);
        Item := FileItems[param_num_int];
        if Item^.fi_Signature_Sub <> '' then
        begin
          aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
          if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item^.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
          begin
            if BL_PROCEED_LOGGING then
              Progress.Log(RPad('Proceed' + HYPHEN + 'File Sub-Signature Match:', RPAD_VALUE) + anEntry.EntryName + SPACE + 'Bates:' + IntToStr(anEntry.ID));
            Result := True;
            break;
          end;
        end;
      end
    end
  end;
end;

// -----------------------------------------------------------------------------
// Get Full Name
// -----------------------------------------------------------------------------
function GetFullName(Item: PSQL_FileSearch): string;
var
  ApplicationName: string;
  TypeName: string;
  OSName: string;
begin
  Result := '';
  ApplicationName := Item^.fi_Name_Program;
  TypeName := Item^.fi_Name_Program_Type;
  OSName := Item^.fi_Name_OS;
  if (ApplicationName <> '') then
  begin
    if (TypeName <> '') then
      Result := format('%0:s %1:s', [ApplicationName, TypeName])
    else if (ApplicationName <> '') then
      Result := ApplicationName
  end
  else
    Result := TypeName;
  if OSName <> '' then
    Result := Result + ' ' + OSName;
end;

function GetNodeAttributes(AxmlNode: TXmlNode): string;
var
  attr: TXmlNode;
  x: integer;
begin
  Result := '';
  if AxmlNode = nil then
    Exit;
  for x := 0 to AxmlNode.AttributeCount - 1 do
  begin
    if not Progress.isRunning then
      break;
    attr := AxmlNode.Attributes[x];
    if (attr <> nil) then
      Result := attr.ValueUnicode;
    Result := trim(Result);
  end;
end;

// -----------------------------------------------------------------------------
// Length of Array Table
// -----------------------------------------------------------------------------
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
var
  i: integer;
begin
  Result := 0;
  for i := 1 to 100 do
  begin
    if anArray[i].sql_col = '' then
      break;
    Result := i;
  end;
end;

{$IF DEFINED (ISFEXGUI)}

function MakeButton: boolean;
var
  concat_str: string;
  CreateButton_StringList: TStringList;
  Item: PSQL_FileSearch;
  n: integer;
  next_str: string;
  unit_name_str: string;
begin
  Result := False;
  LoadFileItems;
  CreateButton_StringList := TStringList.Create;
  try
    unit_name_str := StringReplace(PROGRAM_NAME, ' ', '_', [rfReplaceAll]);
    MakeButtonHeader(PROGRAM_NAME, SCRIPT_NAME, ICON_24_CHAT, CreateButton_StringList);
    concat_str := '';
    for n := 1 to NUMBEROFSEARCHITEMS do
    begin
      Item := FileItems[n];
      current_str := Item^.fi_Name_Program;
      if n <> NUMBEROFSEARCHITEMS then
        next_str := FileItems[n + 1].fi_Name_Program
      else
        next_str := '';
      if current_str <> next_str then
      begin
        concat_str := concat_str + ' ' + IntToStr(n);
        CreateButton_StringList.Add('  Button.AddDropMenu(''' + Item^.fi_Name_Program + ''', GetScriptsDir + ''Artifacts\' + SCRIPT_NAME + ''', ''' + concat_str + ''', ' + IntToStr(Item^.fi_Icon_Program) +
          ', BTNS_SHOWCAPTION or BTNS_DROPDOWN);');
        concat_str := '';
      end
      else
        concat_str := concat_str + ' ' + IntToStr(n);
    end;
    CreateButton_StringList.Add('end;');
    CreateButton_StringList.Add('end.');
  finally
    Progress.Log(CreateButton_StringList.Text);
    if CreateButton_StringList.Count > 0 then
    begin
      CreateButton_StringList.SaveToFile(GetScriptsDir + '\Artifacts\Common\Toolbar\Button_Artifacts_' + unit_name_str + '.pas');
      Result := True;
    end;
    CreateButton_StringList.free;
  end;
end;
{$IFEND}

// -----------------------------------------------------------------------------
// Setup Column For Folder
// -----------------------------------------------------------------------------
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
var
  NumberOfColumns: integer;
  Field: TDataStoreField;
  i: integer;
  column_label: string;
  Item: PSQL_FileSearch;
begin
  Result := True;
  Item := FileItems[aReferenceNumber];
  NumberOfColumns := ColCount;
  setlength(col_DF, ColCount + 1);

  if assigned(anArtifactFolder) then
  begin
    for i := 1 to NumberOfColumns do
    begin
      try
        if not Progress.isRunning then
          Exit;
        Field := gArtifactsDataStore.DataFields.FieldByName(aItems[i].fex_col);
        if assigned(Field) and (Field.FieldType <> aItems[i].col_type) then
        begin
          MessageUser(SCRIPT_NAME + DCR + 'WARNING: New column: ' + DCR + aItems[i].fex_col + DCR + 'already exists as a different type. Creation skipped.');
          Result := False;
        end
        else
        begin
          column_label := '';
          col_DF[i] := gArtifactsDataStore.DataFields.Add(aItems[i].fex_col + column_label, aItems[i].col_type);
          if col_DF[i] = nil then
          begin
            MessageUser(SCRIPT_NAME + DCR + 'Cannot use a fixed field. Please contact support@getdata.com quoting the following error: ' + DCR + SCRIPT_NAME + SPACE + IntToStr(aReferenceNumber) + SPACE + aItems[i].fex_col);
            Result := False;
          end;
        end;
      except
        MessageUser(ATRY_EXCEPT_STR + 'Failed to create column');
      end;
    end;

    // Set the Source Columns --------------------------------------------------
    col_source_file := gArtifactsDataStore.DataFields.GetFieldByName('Source_Name');
    col_source_path := gArtifactsDataStore.DataFields.GetFieldByName('Source_Path');
    col_source_created := gArtifactsDataStore.DataFields.GetFieldByName('Source_Created');
    col_source_modified := gArtifactsDataStore.DataFields.GetFieldByName('Source_Modified');

    // Columns -----------------------------------------------------------------
    if Result then
    begin
      // Enables the change of column headers when switching folders - This is the order of displayed columns
      for i := 1 to NumberOfColumns do
      begin
        if not Progress.isRunning then
          break;
        if aItems[i].Show then
        begin
          // Progress.Log('Add Field Name: ' + col_DF[i].FieldName);
          anArtifactFolder.AddField(col_DF[i]);
        end;
      end;

      if (Item^.fi_Process_As = 'POSTPROCESS') then
        anArtifactFolder.AddField(col_source_path)
      else
      begin
        anArtifactFolder.AddField(col_source_file);
        anArtifactFolder.AddField(col_source_path);
        anArtifactFolder.AddField(col_source_created);
        anArtifactFolder.AddField(col_source_modified);
      end;
    end;
  end;
end;

function TestForDoProcess(ARefNum: integer): boolean;
var
  Item: PSQL_FileSearch;
begin
  Result := False;
  Item := FileItems[ARefNum];
  if Item^.fi_Process_As = 'POSTPROCESS' then
  begin
    Result := True;
    Exit;
  end;
  if (ARefNum <= NUMBEROFSEARCHITEMS) and Progress.isRunning then
  begin
    if (CmdLine.Params.Indexof(IntToStr(ARefNum)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      Progress.Log(RPad('Process List #' + IntToStr(ARefNum) + SPACE + '(' + IntToStr(gArr_ValidatedFiles_TList[ARefNum].Count) + '):', RPAD_VALUE) + Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type + RUNNING);
      Result := True;
    end;
  end
{$IF DEFINED (ISFEXGUI)}
  else
  begin
    if not Progress.isRunning then
      Exit;
    Progress.Log('Error: RefNum > NUMBEROFSEARCHITEMS'); // noslz
  end;
{$ELSE}
    ;
{$IFEND}
end;

function TotalValidatedFileCountInTLists: integer;
var
  i: integer;
begin
  Result := 0;
  for i := 1 to NUMBEROFSEARCHITEMS do
  begin
    if not Progress.isRunning then
      break;
    Result := Result + gArr_ValidatedFiles_TList[i].Count;
  end;
end;

function TestForASCII(str: string): boolean var c: char;
begin
  Result := True;
  if trim(str) <> '' then
  begin
    for c in str do
    begin
      if not Progress.isRunning then
        break;
      if not InRange(Ord(c), $0020, $007F) then
      begin
        Result := False;
        Exit;
      end;
    end;
  end;
end;

procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; Reference_Number: integer; aItems: TSQL_Table_array);
const
  FORMAT_STR = '%-4s %-80s %-8s %-10s';
var
  aArtifactEntry: TEntry;
  aArtifactEntryParent: TEntry;
  ADDList: TList;
  col_DF: TDataStoreFieldArray;
  ColCount: integer;
  Display_Name_str: string;
  DNT_sql_col: string;
  EventData_StringList: TStringList;
  Evtx: TEvtx;
  evtx_StringList: TStringList;
  g, i, j, k: integer;
  Item: PSQL_FileSearch;
  newEntryReader: TEntryReader;
  node_attrib_str: string;
  records_read_int: integer;
  tmp_str: string;
  variant_Array: array of variant;
  xmlDoc: TNativeXml;
  xmlNode: TXmlNode;

  procedure NullTheArray;
  var
    n: integer;
  begin
    for n := 1 to ColCount do
      variant_Array[n] := null;
  end;

  procedure AddToColumns(evtx_col_nam_str: string; evtx_col_val_str: string);
  var
    column_val_dt: TDateTime;
    column_val_int: integer;
    kw_description_str: string;
    lv_description_str: string;
    lv_description_int: integer;
  begin
    for g := 1 to ColCount do
    begin
      if not Progress.isRunning then
        break;
      DNT_sql_col := aItems[g].sql_col;
      Delete(DNT_sql_col, 1, 4);

      // Change EventRecordID to an integer
      if (evtx_col_nam_str = 'EventRecordID') and (DNT_sql_col = evtx_col_nam_str) then // noslz
      begin
        try
          column_val_int := StrToInt(evtx_col_val_str);
          variant_Array[g] := column_val_int;
        except
          Progress.Log(ATRY_EXCEPT_STR + 'converting evtx EventRecordID to int' + HYPHEN + 'row: ' + IntToStr(j) + ')'); // noslz
        end;
        Exit;
      end;

      // Change Event Level Description
      if (evtx_col_nam_str = 'Level') and (DNT_sql_col = evtx_col_nam_str) then // noslz
      begin
        lv_description_str := '';
        try
          lv_description_int := StrToInt(trim(evtx_col_val_str));
          lv_description_str := LevelDescription(lv_description_int);
        except
          Progress.Log(ATRY_EXCEPT_STR + 'converting level description int' + HYPHEN + 'row: ' + IntToStr(j) + ')'); // noslz
        end;

        if lv_description_str = '' then
          variant_Array[g] := evtx_col_val_str
        else
          variant_Array[g] := lv_description_str;
        Exit;
      end;

      // Change Keyword
      if (evtx_col_nam_str = 'Keywords') and (DNT_sql_col = evtx_col_nam_str) then // noslz
      begin
        kw_description_str := KeyWordDescription(evtx_col_val_str);
        if kw_description_str = '' then
          variant_Array[g] := evtx_col_val_str
        else
          variant_Array[g] := kw_description_str;
        Exit;
      end;

      // Change TimeCreated to a DateTime
      if (evtx_col_nam_str = 'TimeCreated') and (DNT_sql_col = evtx_col_nam_str) then // noslz
      begin
        evtx_col_val_str := StringReplace(evtx_col_val_str, 'SystemTime=', '', [rfReplaceAll]); // noslz
        evtx_col_val_str := StringReplace(evtx_col_val_str, '"', '', [rfReplaceAll]); // noslz
        try
          column_val_dt := StrToDateTime(evtx_col_val_str);
          variant_Array[g] := column_val_dt;
        except
          Progress.Log(ATRY_EXCEPT_STR + 'converting evtx date/time' + HYPHEN + 'row: ' + IntToStr(j) + ')'); // noslz
        end;
        Exit;
      end;

      // EventData
      if (evtx_col_nam_str = 'EventData') and (DNT_sql_col = evtx_col_nam_str) then // noslz
      begin
        try
          variant_Array[g] := trim(evtx_col_val_str);
        except
          Progress.Log(ATRY_EXCEPT_STR + 'converting EventData' + HYPHEN + 'row: ' + IntToStr(j) + ')'); // noslz
        end;
        Exit;
      end;

      // Process the String columns
      if DNT_sql_col = evtx_col_nam_str then
      begin
        variant_Array[g] := evtx_col_val_str;
        Exit;
      end;
    end;
  end;

  procedure AddToModule;
  var
    NEntry: TArtifactItem;
    populate_bl, IsAllEmpty, IsAllNull: boolean;
  begin
    populate_bl := False;
    IsAllEmpty := True;
    IsAllNull := True;

    for g := 1 to ColCount do
    begin
      if not VarIsEmpty(variant_Array[g]) then
      begin
        IsAllEmpty := False;
        break;
      end;
    end;

    for g := 1 to ColCount do
    begin
      if not VarIsNull(variant_Array[g]) then
      begin
        IsAllNull := False;
        break;
      end;
    end;

    if (not IsAllNull) and (not IsAllEmpty) then
    begin
      NEntry := TArtifactItem.Create;
      NEntry.SourceEntry := aArtifactEntry;
      NEntry.Parent := anArtifactFolder;
      NEntry.PhysicalSize := 0;
      NEntry.LogicalSize := 0;

      // Populate the columns
      for g := 1 to ColCount do
      begin
        if not Progress.isRunning then
          break;
        if (not VarIsNull(variant_Array[g])) and (not VarIsEmpty(variant_Array[g])) then
          try
            if (col_DF[g].FieldType = ftDateTime) then
              col_DF[g].AsDateTime[NEntry] := variant_Array[g];
            if (col_DF[g].FieldType = ftInteger) then
              col_DF[g].AsInteger[NEntry] := variant_Array[g];
            if (col_DF[g].FieldType = ftLargeInt) then
              col_DF[g].AsInt64[NEntry] := variant_Array[g];
            if (col_DF[g].FieldType = ftString) and VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
              col_DF[g].AsString(NEntry) := variant_Array[g];
            if (col_DF[g].FieldType = ftBytes) then
              col_DF[g].AsBytes[NEntry] := variantToArrayBytes(variant_Array[g]);
          except
            on e: exception do
            begin
              Progress.Log(e.message);
              Progress.Log(ATRY_EXCEPT_STR + 'Error adding to columns in Procedure AddToModule');
            end;
          end;
      end;
      ADDList.Add(NEntry);
    end;
    NullTheArray;
  end;

// Start of Do Process ---------------------------------------------------------
begin
  Item := FileItems[Reference_Number];
  temp_process_counter := 0;
  ColCount := LengthArrayTABLE(aItems);
  if (gArr_ValidatedFiles_TList[Reference_Number].Count > 0) then
  begin
    if assigned(anArtifactFolder) then
    begin
      process_proceed_bl := True;
      SetUpColumnforFolder(Reference_Number, anArtifactFolder, col_DF, ColCount, aItems);
      if process_proceed_bl then
      begin
        setlength(variant_Array, ColCount + 1);
        ADDList := TList.Create;
        newEntryReader := TEntryReader.Create;
        try
          Progress.Max := gArr_ValidatedFiles_TList[Reference_Number].Count;
          Progress.DisplayMessageNow := 'Process' + SPACE + PROGRAM_NAME + ' - ' + Item^.fi_Name_OS + RUNNING;
          Progress.CurrentPosition := 1;

          // Loop Validated Files ----------------------------------------------
          for i := 0 to gArr_ValidatedFiles_TList[Reference_Number].Count - 1 do { addList is freed a the end of this loop }
          begin
            if not Progress.isRunning then
              break;
            Progress.IncCurrentprogress;
            temp_process_counter := temp_process_counter + 1;
            Display_Name_str := GetFullName(Item);
            Progress.DisplayMessageNow := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[Reference_Number].Count) + ')' + RUNNING;
            aArtifactEntry := TEntry(gArr_ValidatedFiles_TList[Reference_Number].items[i]);
            if assigned(aArtifactEntry.Parent) then
              aArtifactEntryParent := TEntry(aArtifactEntry.Parent);

            if assigned(aArtifactEntry) and newEntryReader.opendata(aArtifactEntry) and (newEntryReader.Size > 0) and Progress.isRunning then
            begin
              if USE_FLAGS_BL then
                aArtifactEntry.Flags := aArtifactEntry.Flags + [Flag8]; // Gray Flag = Process Routine

              // ----------------------------------------------------------------
              // PROCESS_AS_EVENTLOG
              // ----------------------------------------------------------------
              if UpperCase(Item^.fi_Process_As) = PROCESS_AS_EVENTLOG then
              begin
                evtx_StringList := TStringList.Create;
                try
                  if newEntryReader.opendata(aArtifactEntry) then
                  begin
                    newEntryReader.Position := 0;
                    Evtx := TEvtx.Create(newEntryReader);
                    try
                      try
                        if assigned(Evtx) then
                        try
                          Evtx.Open;
                        except
                          Progress.Log('Evt.Open exception');
                          Exit;
                        end
                        else
                        begin
                          Progress.Log('Evtx.Open: Not assigned Evtx'); // noslz
                          Exit;
                        end;
                      except
                        Progress.Log(ATRY_EXCEPT_STR + 'Evtx.Open error' + HYPHEN + aArtifactEntry.EntryName + SPACE + 'Bates: ' + IntToStr(aArtifactEntry.ID));
                        Exit;
                      end;
                      if not assigned(Evtx) then
                      begin
                        Progress.Log(ATRY_EXCEPT_STR + 'Evtx not assigned' + HYPHEN + aArtifactEntry.EntryName + SPACE + 'Bates: ' + IntToStr(aArtifactEntry.ID));
                        Exit;
                      end;
                      Progress.Log(format('%-4s %-90s %-8s %-10s', [IntToStr(i + 1) + '.', aArtifactEntry.EntryName, IntToStr(Evtx.RowCount), '(row count)']));

                      if assigned(Evtx) and (Evtx.RowCount > 0) then
                      begin

                        // -----------------------------------------------------
                        // Collect Row data
                        // -----------------------------------------------------
                        Progress.Initialize(Evtx.RowCount, 'Processing ' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gtotal_validated_files_int - 1) + ':' + SPACE + aArtifactEntry.EntryName);

                        for j := 0 to Evtx.RowCount - 1 do // j is the Event Log row
                        begin
                          if not Progress.isRunning then
                            break;
                          NullTheArray;
                          xmlDoc := TNativeXml.Create(nil);
                          try

                            // Test the row
                            try
                              xmlDoc := Evtx.ReadRow(j);
                              if assigned(xmlDoc) and (xmlDoc.RootNodeCount > 0) and (not xmlDoc.IsEmpty) then
                              begin
                                xmlDoc.EolStyle := esCRLF;
                                xmlDoc.XmlFormat := xfReadable;
                                xmlDoc.IndentString := '    ';
                              end
                              else
                              begin
                                if not assigned(xmlDoc) then
                                  Progress.Log(format(FORMAT_STR, ['', HYPHEN + 'Row' + SPACE + IntToStr(j) + ': is not assigned', '', '']))
                                else if xmlDoc.RootNodeCount = 0 then
                                  Progress.Log(format(FORMAT_STR, ['', HYPHEN + 'Row' + SPACE + IntToStr(j) + ': xmlDoc.RootNodeCount = 0', '', ''])) // noslz
                                else if xmlDoc.IsEmpty then
                                  Progress.Log(format(FORMAT_STR, ['', HYPHEN + 'Row' + SPACE + IntToStr(j) + ': xlmDoc is empty.', '', ''])); // noslz
                                Continue;
                              end;
                            except
                              Progress.Log(ATRY_EXCEPT_STR + 'Evtx.ReadRow(j)'); // noslz
                              Continue;
                            end;

                            // Skip delete rows that can contain invalid char
                            if Evtx.isCurrentRowDeleted then
                            begin
                              //Progress.Log(format(FORMAT_STR, ['', HYPHEN + 'Row' + SPACE + IntToStr(j) + ': is deleted.', '', ''])); // noslz
                              Continue;
                            end;

                            if assigned(xmlDoc) and (xmlDoc.RootNodeCount > 0) and (not xmlDoc.IsEmpty) then
                            begin
                              for k := 0 to xmlDoc.root.ElementCount - 1 do
                              begin
                                if not Progress.isRunning then
                                  break;

                                // System ======================================
                                if xmlDoc.root.Elements[k].Name = 'System' then // noslz
                                begin
                                  xmlNode := xmlDoc.root.Elements[k];
                                  if xmlNode.ElementCount > 0 then
                                  begin
                                    for n := 0 to xmlNode.ElementCount - 1 do
                                    begin
                                      if not Progress.isRunning then
                                        break;
                                      if xmlNode.Elements[n].ValueUnicode = '' then
                                      begin
                                        node_attrib_str := GetNodeAttributes(xmlNode.Elements[n]);
                                        AddToColumns(xmlNode.Elements[n].Name, node_attrib_str);
                                        // Progress.Log(RPad(IntToStr(n) + '. ' + HYPHEN + xmlNode.Elements[n].Name, RPAD_VALUE) + node_attrib_str);
                                      end
                                      else
                                      begin
                                        if TestForASCII(xmlNode.Elements[n].ValueUnicode) = False then
                                        begin
                                          // AddToColumns(xmlNode.Elements[n].Name, '');
                                          // Progress.Log(RPad(IntToStr(n) + '. ' + HYPHEN + xmlNode.Elements[n].Name, RPAD_VALUE) + 'CORRUPT_RESULT_1');
                                        end
                                        else
                                        begin
                                          AddToColumns(xmlNode.Elements[n].Name, xmlNode.Elements[n].ValueUnicode);
                                          // Progress.Log(RPad(IntToStr(n) + '. ' + HYPHEN + xmlNode.Elements[n].Name, RPAD_VALUE) + xmlNode.Elements[n].ValueUnicode);
                                        end;
                                      end;
                                    end;
                                  end;
                                end;

                                // Event Data ==================================
                                if xmlDoc.root.Elements[k].Name = 'EventData' then // noslz
                                begin
                                  EventData_StringList := TStringList.Create;
                                  try
                                    xmlNode := xmlDoc.root.Elements[k];
                                    if xmlNode.ElementCount > 0 then
                                    begin
                                      for n := 0 to xmlNode.ElementCount - 1 do
                                      begin
                                        if not Progress.isRunning then
                                          break;
                                        begin
                                          if TestForASCII(xmlNode.Elements[n].ValueUnicode) = False then
                                          begin
                                          // Progress.Log('Row:' + SPACE + IntToStr(j) + ': ' + aArtifactEntry.EntryName + ':' + SPACE + xmlNode.Name + ':' + SPACE + 'NOT ASCII'); // noslz
                                          end
                                          else
                                          begin
                                            node_attrib_str := GetNodeAttributes(xmlNode.Elements[n]);
                                            tmp_str := node_attrib_str + ': ' + xmlNode.Elements[n].ValueUnicode;
                                            EventData_StringList.Add(tmp_str);
                                          end;
                                        end;
                                      end;
                                    end;
                                    AddToColumns(xmlNode.Name, EventData_StringList.CommaText);
                                  finally
                                    EventData_StringList.free;
                                  end;
                                end;

                              end;
                              if BL_LOG_XML_DATA then
                              begin
                                evtx_StringList.Text := xmlDoc.WriteToString;
                                Progress.Log('Row ' + IntToStr(j) + ': ' + aArtifactEntry.EntryName + SPACE + '(Bates: ' + IntToStr(aArtifactEntry.ID) + ')');
                                Progress.Log(evtx_StringList.Text);
                                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                              end;
                            end;
                            Progress.IncCurrentprogress;
                            AddToModule;
                          finally
                            if assigned(xmlDoc) then
                              xmlDoc.free;
                          end;
                        end;
                        if assigned(Evtx) then
                          Evtx.Close;
                      end;
                    finally
                      if assigned(Evtx) then
                        Evtx.free;
                    end;
                  end;
                finally
                  evtx_StringList.free;
                end;
              end;
            end;
            // Progress.Log(StringOfChar('-', CHAR_LENGTH));
            Progress.IncCurrentprogress;
          end;

          // Add to the gArtifactsDataStore
          if assigned(ADDList) and Progress.isRunning then
          begin
            gArtifactsDataStore.Add(ADDList);
            Progress.Log(RPad('Total Artifacts:', RPAD_VALUE) + IntToStr(ADDList.Count));

            // Export L01 files where artifacts are found (controlled by Artifact_Utils.pas)
            if (gArr_ValidatedFiles_TList[Reference_Number].Count > 0) and (ADDList.Count > 0) then
              ExportToL01(gArr_ValidatedFiles_TList[Reference_Number], Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type);

            ADDList.Clear;
          end;

          Progress.Log(StringOfChar('-', CHAR_LENGTH));
        finally
          records_read_int := 0;
          FreeAndNil(ADDList);
          FreeAndNil(newEntryReader);
        end;
      end;

    end
    else
      Progress.Log(RPad('', RPAD_VALUE) + 'IMPORTANT: Process failed. Not assigned a tree folder.');

  end
  else
  begin
    Progress.Log(RPad('', RPAD_VALUE) + 'No files to process.');
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
  end;

end;

// ==========================================================================================================================================================
// Start of Script
// ==========================================================================================================================================================
const
  SEARCHING_FOR_ARTIFACT_FILES_STR = 'Searching for Artifact files';

var
  AboutToProcess_StringList: TStringList;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  aFolderEntry: TEntry;
  AllFoundListUnique: TUniqueListOfEntries;
  AllFoundList_count: integer;
  DeleteFolder_display_str: string;
  DeleteFolder_TList: TList;
  Display_StringList: TStringList;
  Enum: TEntryEnumerator;
  ExistingFolders_TList: TList;
  FieldItunesDomain: TDataStoreField;
  FieldItunesName: TDataStoreField;
  FindEntries_StringList: TStringList;
  FoundList, AllFoundList: TList;
  i: integer;
  Item: PSQL_FileSearch;
  iTunes_Domain_str: string;
  iTunes_Name_str: string;
  ResultInt: integer;

{$IF DEFINED (ISFEXGUI)}
  MyScriptForm: TScriptForm;
{$IFEND}

procedure LogExistingFolders(ExistingFolders_TList: TList; DeleteFolder_TList: TList); // Compare About to Process Folders with Existing Artifact Module Folders
const
  LOG_BL = False;
var
  aFolderEntry: TEntry;
  s, t: integer;
begin
  if LOG_BL then
    Progress.Log('Existing Folders:');
  for s := 0 to ExistingFolders_TList.Count - 1 do
  begin
    if not Progress.isRunning then
      break;
    aFolderEntry := (TEntry(ExistingFolders_TList[s]));
    if LOG_BL then
      Progress.Log(HYPHEN + aFolderEntry.FullPathName);
    for t := 0 to AboutToProcess_StringList.Count - 1 do
    begin
      if aFolderEntry.FullPathName = AboutToProcess_StringList[t] then
        DeleteFolder_TList.Add(aFolderEntry);
    end;
  end;
  if LOG_BL then
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
end;

// Start of script
begin
  LoadFileItems;
  Progress.Log(SCRIPT_NAME + ' started.');
  Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME;
  Progress.LogType := ltVerbose; // ltOff, ltVerbose, ltDebug, ltTechnical

  if not StartingChecks then
  begin
    Progress.DisplayMessageNow := ('Processing complete.');
    Exit;
  end;

  param := '';
  test_param_int := 0;
  process_all_bl := False;

  gParameter_Num_StringList := TStringList.Create;
  try { gParameter_Num_StringList }
    Progress.Log(IntToStr(CmdLine.ParamCount) + ' processing parameters received: ');
    if (CmdLine.ParamCount > 0) then
    begin
      for n := 0 to CmdLine.ParamCount - 1 do
      begin
        if not Progress.isRunning then
          break;

        param := param + '"' + CmdLine.Params[n] + '"' + ' ';
        // Validate Parameters
        if (RegexMatch(CmdLine.Params[n], '\d{1,2}$', False)) then
        begin
          try
            test_param_int := StrToInt(CmdLine.Params[n]);
            if (test_param_int <= 0) or (test_param_int > NUMBEROFSEARCHITEMS) then
            begin
              MessageUser('Invalid parameter received: ' + (CmdLine.Params[n]) + #13#10 + ('Maximum is: ' + IntToStr(NUMBEROFSEARCHITEMS) + DCR + SCRIPT_NAME + ' will terminate.'));
              Exit;
            end;
            gParameter_Num_StringList.Add(CmdLine.Params[n]);
            Item := FileItems[test_param_int];
            Progress.Log(RPad(HYPHEN + 'Param ' + IntToStr(n) + ' = ' + CmdLine.Params[n], RPAD_VALUE) + format('%-10s %-10s %-25s %-12s', ['Ref#: ' + CmdLine.Params[n], CATEGORY_NAME,
              Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type, Item^.fi_Name_OS]));
          except
            begin
              Progress.Log(ATRY_EXCEPT_STR + 'Error validating parameter. ' + SCRIPT_NAME + ' will terminate.');
              Exit;
            end;
          end;
        end;
      end;
      trim(param);
    end
    else
    begin
      MessageUser('No parameters received.');
      Exit;
    end;
    Progress.Log(StringOfChar('-', CHAR_LENGTH));

    // Progress Bar Text
    progress_program_str := '';
    if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      progress_program_str := PROGRAM_NAME;
      process_all_bl := True;
    end;

    for n := 1 to NUMBEROFSEARCHITEMS do
    begin
      if not Progress.isRunning then
        break;
      if (CmdLine.Params.Indexof(IntToStr(n)) > -1) then
      begin
        Item := FileItems[n];
        progress_program_str := 'Ref#:' + SPACE + param + SPACE + Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type;
        break;
      end;
    end;

    if progress_program_str = '' then
      progress_program_str := 'Other';

    gArtifactsDataStore := GetDataStore(DATASTORE_ARTIFACTS);
    if not assigned(gArtifactsDataStore) then
    begin
      Progress.Log(DATASTORE_ARTIFACTS + ' module not located.' + DCR + TSWT);
      Exit;
    end;

    try { gArtifactsDataStore }
      gFileSystemDataStore := GetDataStore(DATASTORE_FILESYSTEM);
      if not assigned(gFileSystemDataStore) then
      begin
        Progress.Log(DATASTORE_FILESYSTEM + ' module not located.' + DCR + TSWT);
        Exit;
      end;

      try { gFileSystemDataStore }
        FieldItunesDomain := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_DOMAIN);
        FieldItunesName := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_NAME);

        // Create TLists For Valid Files
        for n := 1 to NUMBEROFSEARCHITEMS do
        begin
          if not Progress.isRunning then
            break;
          gArr_ValidatedFiles_TList[n] := TList.Create;
        end;

        try { gArr_ValidatedFiles_TList }
          AboutToProcess_StringList := TStringList.Create;
          AboutToProcess_StringList.Sorted := True; // Must be sorted or duplicates in the following line has no effect
          AboutToProcess_StringList.Duplicates := dupIgnore;
          try
            Display_str := '';
            previous_str := '';
            current_str := '';
            if (CmdLine.Params.Indexof('MASTER') = -1) then
            begin
              if (CmdLine.Params.Indexof('NOSHOW') = -1) then
              begin
                Display_StringList := TStringList.Create;
                Display_StringList.Sorted := True; // Must be sorted or duplicates in the following line has no effect
                Display_StringList.Duplicates := dupIgnore;
                try
                  // Process All - Create the AboutToProcess_StringList and the Message Box text
                  if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
                  begin
                    previous_str := '';
                    current_str := '';
                    for n := 1 to NUMBEROFSEARCHITEMS do
                    begin
                      if not Progress.isRunning then
                        break;
                      Item := FileItems[n];
                      current_str := GetFullName(Item);
                      AboutToProcess_StringList.Add(CATEGORY_NAME + BS + current_str);
                      if current_str <> previous_str then
                      begin
                        Display_StringList.Add(current_str);
                      end;
                      previous_str := current_str;
                    end;
                    Display_str := Display_StringList.Text;
                  end
                  else
                  begin
                    // Process Individual - Create the AboutToProcess_StringList and the Message Box text
                    if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
                    begin
                      for n := 0 to gParameter_Num_StringList.Count - 1 do
                      begin
                        if not Progress.isRunning then
                          break;
                        temp_int := StrToInt(gParameter_Num_StringList[n]);
                        Item := FileItems[temp_int];
                        AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(Item));
                      end;
                      Display_str := AboutToProcess_StringList.Text;
                    end;
                  end;
                finally
                  if (Display_StringList.Count > 0) then
                  begin
                    Progress.Log('Extract Artifacts:');
                    Progress.Log(Display_StringList.Text);
                  end;
                  Display_StringList.free;
                end;

                // // Log About to Process
                // Progress.Log('About to Process:');
                // for s := 0 to AboutToProcess_StringList.Count - 1 do
                // Progress.Log(AboutToProcess_StringList[s]);
                // Progress.Log(StringOfChar('-', CHAR_LENGTH));

                // Show the form
                ResultInt := 1; // Continue AboutToProcess

{$IF DEFINED (ISFEXGUI)}
                if AboutToProcess_StringList.Count > 30 then
                begin
                  MyScriptForm := TScriptForm.Create;
                  try
                    MyScriptForm.SetCaption(SCRIPT_DESCRIPTION);
                    MyScriptForm.SetText(trim(Display_str));
                    ResultInt := idCancel;
                    if MyScriptForm.ShowModal then
                      ResultInt := idOk
                  finally
                    FreeAndNil(MyScriptForm);
                  end;
                end
                else
                begin
                  ResultInt := MessageBox('Extract Artifacts?' + DCR + Display_str, 'Extract Artifacts' + HYPHEN + PROGRAM_NAME, (MB_OKCANCEL or MB_ICONQUESTION or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                end;
{$IFEND}
              end;

              if ResultInt > 1 then // User cancels
              begin
                Progress.Log(CANCELED_BY_USER);
                Progress.DisplayMessageNow := CANCELED_BY_USER;
                Exit;
              end;

            end; { About to process }

            // Deal with Existing Artifact Folders
            ExistingFolders_TList := TList.Create;
            DeleteFolder_TList := TList.Create;
            try
              if gArtifactsDataStore.Count > 1 then
              begin
                anEntry := gArtifactsDataStore.First;
                while assigned(anEntry) and Progress.isRunning do
                begin
                  if anEntry.isDirectory then
                    ExistingFolders_TList.Add(anEntry);
                  anEntry := gArtifactsDataStore.Next;
                end;
                gArtifactsDataStore.Close;
              end;

              LogExistingFolders(ExistingFolders_TList, DeleteFolder_TList);

              // Create the delete folder TList and display string
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                for s := 0 to DeleteFolder_TList.Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  aFolderEntry := (TEntry(DeleteFolder_TList[s]));
                  DeleteFolder_display_str := DeleteFolder_display_str + #13#10 + aFolderEntry.FullPathName;
                  Progress.Log(RPad('Replace this Artifact folder?:', RPAD_VALUE) + aFolderEntry.FullPathName);
                end;
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
              end;

              // Message Box - When artifact folder is already present ---------------------
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                if (CmdLine.Params.Indexof('MASTER') = -1) then
                begin
{$IF DEFINED (ISFEXGUI)}
                  if (CmdLine.Params.Indexof('NOSHOW') = -1) then
                  begin
                    if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
                    begin
                      ResultInt := MessageBox('Artifacts have already been processed:' + #13#10 + DeleteFolder_display_str + DCR + 'Replace the existing Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME,
                        (MB_OKCANCEL or MB_ICONWARNING or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                    end;
                  end
                  else
                    ResultInt := idOk;
                  case ResultInt of
                    idOk:
                      begin
                        try
                          gArtifactsDataStore.Remove(DeleteFolder_TList);
                        except
                          MessageBox('ERROR: There was an error deleting existing artifacts.' + #13#10 + 'Save then reopen your case.', SCRIPT_NAME, (MB_OK or MB_ICONINFORMATION or MB_SETFOREGROUND or MB_TOPMOST));
                        end;
                        Progress.Log(RPad('Replace Existing Artifacts:', RPAD_VALUE) + 'True');
                      end;
                    idCancel:
                      begin
                        Progress.Log(CANCELED_BY_USER);
                        Progress.DisplayMessageNow := CANCELED_BY_USER;
                        Exit;
                      end;
                  end;
{$IFEND}
                end;
              end;
            finally
              ExistingFolders_TList.free;
              DeleteFolder_TList.free;
            end;
          finally
            AboutToProcess_StringList.free;
          end;

          // Create the RegEx Search String
          regex_search_str := '';
          begin
            for n := 1 to NUMBEROFSEARCHITEMS do
            begin
              if not Progress.isRunning then
                break;
              Item := FileItems[n];
              if (CmdLine.Params.Indexof(IntToStr(n)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                if Item^.fi_Regex_Search <> '' then
                  regex_search_str := regex_search_str + '|' + Item^.fi_Regex_Search;
                if Item^.fi_Rgx_Itun_Bkup_Dmn <> '' then
                  regex_search_str := regex_search_str + '|' + Item^.fi_Rgx_Itun_Bkup_Dmn;
                if Item^.fi_Rgx_Itun_Bkup_Nme <> '' then
                  regex_search_str := regex_search_str + '|' + Item^.fi_Rgx_Itun_Bkup_Nme;
              end;
            end;
          end;
          if (regex_search_str <> '') and (regex_search_str[1] = '|') then
            Delete(regex_search_str, 1, 1);

          AllFoundList := TList.Create;
          try
            AllFoundListUnique := TUniqueListOfEntries.Create;
            FoundList := TList.Create;
            FindEntries_StringList := TStringList.Create;
            try
              FindEntries_StringList.Add('**\*.evtx');

              // Find the files by path and add to AllFoundListUnique
              Progress.Initialize(FindEntries_StringList.Count, STR_FILES_BY_PATH + RUNNING);
              Progress.Log('Find files by path' + RUNNING);
              gtick_foundlist_i64 := GetTickCount;
              for i := 0 to FindEntries_StringList.Count - 1 do
              begin
                if not Progress.isRunning then
                  break;
                try
                  Find_Entries_By_Path(gFileSystemDataStore, FindEntries_StringList[i], FoundList, AllFoundListUnique);
                except
                  Progress.Log(RPad(ATRY_EXCEPT_STR, RPAD_VALUE) + 'Find_Entries_By_Path');
                end;
                Progress.IncCurrentprogress;
              end;

              Progress.Log(StringOfChar('-', CHAR_LENGTH));
              Progress.Log(RPad(STR_FILES_BY_PATH + SPACE + '(Unique)' + COLON, RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Add > 40 char files with no extension to Unique List (possible iTunes backup files)

              // Move the AllFoundListUnique list into a TList
              if assigned(AllFoundListUnique) and (AllFoundListUnique.Count > 0) then
              begin
                Enum := AllFoundListUnique.GetEnumerator;
                while Enum.MoveNext do
                begin
                  anEntry := Enum.Current;
                  AllFoundList.Add(anEntry);
                end;
              end;

              // Now work with the TList from now on
              if assigned(AllFoundList) and (AllFoundList.Count > 0) then
              begin
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                Progress.Log('Unique Files by path: ' + IntToStr(AllFoundListUnique.Count));
                Progress.Log(StringOfChar('-', CHAR_LENGTH));

                // Add flags
                if USE_FLAGS_BL then
                begin
                  Progress.Log('Adding flags' + RUNNING);
                  for i := 0 to AllFoundList.Count - 1 do
                  begin
                    if not Progress.isRunning then
                      break;
                    anEntry := TEntry(AllFoundList[i]);
                    anEntry.Flags := anEntry.Flags + [Flag7];
                  end;
                  Progress.Log('Finished adding flags' + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Determine signature
                if assigned(AllFoundList) and (AllFoundList.Count > 0) then
                  SignatureAnalysis(AllFoundList);

                gtick_foundlist_str := (RPad(STR_FILES_BY_PATH + COLON, RPAD_VALUE) + CalcTimeTaken(gtick_foundlist_i64));
              end;

            finally
              FoundList.free;
              AllFoundListUnique.free;
              FindEntries_StringList.free;
            end;

            // Locate the relevant files in the File System
            if assigned(AllFoundList) and (AllFoundList.Count > 0) then
            begin
              Progress.Log(SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              Progress.Log(format(GFORMAT_STR, ['', 'Action', 'Ref#', 'Bates', 'Signature', 'Filename (trunc)', 'Reason'])); // noslz
              Progress.Log(format(GFORMAT_STR, ['', '------', '----', '-----', '---------', '----------------', '------'])); // noslz

              AllFoundList_count := AllFoundList.Count;
              Progress.Initialize(AllFoundList_count, SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              for i := 0 to AllFoundList.Count - 1 do
              begin
                Progress.DisplayMessages := SEARCHING_FOR_ARTIFACT_FILES_STR + SPACE + '(' + IntToStr(i) + ' of ' + IntToStr(AllFoundList_count) + ')' + RUNNING;
                if not Progress.isRunning then
                  break;
                anEntry := TEntry(AllFoundList[i]);

                if (i mod 10000 = 0) and (i > 0) then
                begin
                  //Progress.Log('Processing: ' + IntToStr(i) + ' of ' + IntToStr(AllFoundList.Count) + RUNNING);
                  //Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Set the iTunes Domain String
                iTunes_Domain_str := '';
                if assigned(FieldItunesDomain) then
                begin
                  try
                    iTunes_Domain_str := FieldItunesDomain.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Domain string');
                  end;
                end;

                // Set the iTunes Name String
                iTunes_Name_str := '';
                if assigned(FieldItunesName) then
                begin
                  try
                    iTunes_Name_str := FieldItunesName.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Name string');
                  end;
                end;

                // Run the match
                aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
                if
                (anEntry.Extension <> 'db-shm') and
                (anEntry.Extension <> 'db-wal') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite WAL') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite SHM') then
                begin
                  if RegexMatch(anEntry.EntryName, regex_search_str, False) or
                  RegexMatch(anEntry.FullPathName, regex_search_str, False) or
                  FileSubSignatureMatch(anEntry) or
                  RegexMatch(iTunes_Domain_str, regex_search_str, False) or
                  RegexMatch(iTunes_Name_str, regex_search_str, False) then
                  begin

                    // Sub Signature Match
                    if FileSubSignatureMatch(anEntry) then
                    begin
                      if USE_FLAGS_BL then
                        anEntry.Flags := anEntry.Flags + [Flag2]; // Blue Flag
                      DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                    end
                    else

                    // Regex Name Match
                    begin
                      // Regex Name Match
                      if ((not anEntry.isDirectory) or (anEntry.isDevice)) and (anEntry.LogicalSize > 0) and (anEntry.PhysicalSize > 0) then
                      begin
                        if FileNameRegexSearch(anEntry, iTunes_Domain_str, iTunes_Name_str, regex_search_str) then
                        begin
                          if USE_FLAGS_BL then
                            anEntry.Flags := anEntry.Flags + [Flag1]; // Red Flag
                          DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                        end;
                      end;
                    end;

                  end;
                  Progress.IncCurrentprogress;
                end;
              end;
            end;
          finally
            AllFoundList.free;
          end;

          // Check to see if files were found
          if (TotalValidatedFileCountInTLists = 0) then
          begin
            Progress.Log('No ' + PROGRAM_NAME + SPACE + 'files were found.');
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Not found';
            Exit;
          end
          else
          begin
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
            Progress.Log(RPad('Total Validated Files:', RPAD_VALUE) + IntToStr(TotalValidatedFileCountInTLists));
            Progress.Log(StringOfChar('=', CHAR_LENGTH + 80));
          end;

          // Display the content of the TLists for further processing
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.Log('Lists available to process' + RUNNING);
            for n := 1 to NUMBEROFSEARCHITEMS do
            begin
              if not Progress.isRunning then
                break;
              Item := FileItems[n];
              if gArr_ValidatedFiles_TList[n].Count > 0 then
              begin
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                Progress.Log(RPad('TList ' + IntToStr(n) + ': ' + Item^.fi_Name_Program + SPACE + Item^.fi_Name_Program_Type + HYPHEN + Item^.fi_Name_OS, RPAD_VALUE) + IntToStr(gArr_ValidatedFiles_TList[n].Count));
                for r := 0 to gArr_ValidatedFiles_TList[n].Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  anEntry := (TEntry(TList(gArr_ValidatedFiles_TList[n]).items[r]));
                  if not(Item^.fi_Process_As = 'POSTPROCESS') then
                    Progress.Log(RPad(' Bates: ' + IntToStr(anEntry.ID), RPAD_VALUE) + anEntry.EntryName + SPACE + Item^.fi_Process_As);
                end;
              end;
            end;
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
          end;

          // *** CREATE GUI ***
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Found';

            // Create the Category Folder to the tree ----------------------------------
            gArtConnect_CatFldr := AddArtifactCategory(nil, CATEGORY_NAME, -1, FileItems[1].fi_Icon_Category); { Sort index, icon }
            gArtConnect_CatFldr.Status := gArtConnect_CatFldr.Status + [dstUserCreated];

            if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
            begin
              for n := 1 to NUMBEROFSEARCHITEMS do
              begin
                if not Progress.isRunning then
                  break;

                Item := FileItems[n];
                if (gArr_ValidatedFiles_TList[n].Count > 0) or (Item^.fi_Process_As = 'POSTPROCESS') then
                begin
                  gArtConnect_ProgFldr[n] := AddArtifactConnect(TEntry(gArtConnect_CatFldr),
                  Item^.fi_Name_Program,
                  Item^.fi_Name_Program_Type,
                  Item^.fi_Name_OS,
                  Item^.fi_Icon_Program,
                  Item^.fi_Icon_OS);
                  gArtConnect_ProgFldr[n].Status := gArtConnect_ProgFldr[n].Status + [dstUserCreated];
                end;
              end;
            end
            else
            begin
              if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) or (Item^.fi_Process_As = 'POSTPROCESS') then
              begin
                for n := 0 to gParameter_Num_StringList.Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  temp_int := StrToInt(gParameter_Num_StringList[n]); // temp_int becomes the parameter
                  Item := FileItems[temp_int];
                  if gArr_ValidatedFiles_TList[temp_int].Count > 0 then
                  begin
                    gArtConnect_ProgFldr[temp_int] := AddArtifactConnect(TEntry(gArtConnect_CatFldr),
                    Item^.fi_Name_Program,
                    Item^.fi_Name_Program_Type,
                    Item^.fi_Name_OS,
                    Item^.fi_Icon_Program,
                    Item^.fi_Icon_OS);
                    gArtConnect_ProgFldr[temp_int].Status := gArtConnect_ProgFldr[temp_int].Status + [dstUserCreated];
                  end;
                end;
              end;
            end;
            Progress.Log('Process Lists' + RUNNING);
            Progress.Log(StringOfChar('-', CHAR_LENGTH));

            // *** DO PROCESS ***
            temp_process_counter := 0;
            Progress.CurrentPosition := 0;
            gtotal_validated_files_int := TotalValidatedFileCountInTLists;
            Progress.Max := TotalValidatedFileCountInTLists;
            gtick_doprocess_i64 := GetTickCount;

            Ref_Num := 1; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            Ref_Num := 2; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            Ref_Num := 3; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            Ref_Num := 4; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            Ref_Num := 5; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            Ref_Num := 6; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            Ref_Num := 7; if (TestForDoProcess(Ref_Num)) then DoProcess(gArtConnect_ProgFldr[Ref_Num], Ref_Num, Array_Items_1);
            gtick_doprocess_str := (RPad('DoProcess:', RPAD_VALUE) + CalcTimeTaken(gtick_doprocess_i64));
          end;

          if not Progress.isRunning then
          begin
            Progress.Log(CANCELED_BY_USER);
            Progress.DisplayMessageNow := CANCELED_BY_USER;
          end;

        finally
          for n := 1 to NUMBEROFSEARCHITEMS do
            FreeAndNil(gArr_ValidatedFiles_TList[n]);
        end;

      finally
        if assigned(gFileSystemDataStore) then
          FreeAndNil(gFileSystemDataStore);
      end;

    finally
      if assigned(gArtifactsDataStore) then
        FreeAndNil(gArtifactsDataStore);
    end;

  finally
    if assigned(gParameter_Num_StringList) then
      FreeAndNil(gParameter_Num_StringList);
  end;

  Progress.Log(StringOfChar('-', CHAR_LENGTH));
  Progress.Log(gtick_foundlist_str);
  Progress.Log(gtick_doprocess_str);
  Progress.Log(StringOfChar('-', CHAR_LENGTH));

  // Remove folders with zero
  Sleep(100);
  PostProcess_Remove_Zero(CATEGORY_NAME);
  Sleep(100);
  PostProcess_Remove_L1_Zero(CATEGORY_NAME);

  Progress.Log(SCRIPT_NAME + ' finished.');
  Progress.DisplayMessageNow := ('Processing complete.');

end.
